/* libtunerlimit: adjust when setrlimit() is called
 * Copyright (C) 2009 Kevin Pulo
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Contact:
 * Kevin Pulo
 * kev@pulo.com.au
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>

/* Using _GNU_SOURCE as the manpage suggests can be problematic.
 * We only want the GNU extensions for dlfcn.h, not others like stdio.h.
 * But dlfcn.h checks __USE_GNU, not _GNU_SOURCE.  __USE_GNU is set by
 * features.h when _GNU_SOURCE is set, but features.h has already been loaded
 * by stdio.h, so it doesn't get loaded again.  Ugh.  Setting _GNU_SOURCE
 * earlier drags in other junk (eg. dprintf).  Simplest is just to use
 * __USE_GNU directly.
 */
#define __USE_GNU
#include <dlfcn.h>
#undef __USE_GNU

#include <unistd.h>

#include <sys/time.h>
#include <sys/resource.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>



static int debug = 0;

static const char libtunerlimit_debug_envvar[] = "LIBTUNERLIMIT_DEBUG";

static int do_debug() {
	return (debug || getenv(libtunerlimit_debug_envvar));
}

static void dprintf(char *fmt, ...) {
	va_list ap;
	if (do_debug()) {
		va_start(ap, fmt);
		vfprintf(stderr, fmt, ap);
		va_end(ap);
		fflush(stderr);
	}
}





static void _libtunerlimit_init(void) __attribute__((constructor));
static void _libtunerlimit_fini(void) __attribute__((destructor));

static void _libtunerlimit_init(void) {
	dprintf("libtunerlimit: starting up\n");
}

static void _libtunerlimit_fini(void) {
	dprintf("libtunerlimit: shutting down\n");
}


#define NAMECASE(x) case RLIMIT_##x: return #x;

static const char * resource_name(int resource) {
	switch (resource) {
		NAMECASE(AS)
		NAMECASE(CORE)
		NAMECASE(CPU)
		NAMECASE(DATA)
		NAMECASE(FSIZE)
		NAMECASE(LOCKS)
		NAMECASE(MEMLOCK)
		NAMECASE(MSGQUEUE)
		NAMECASE(NICE)
		NAMECASE(NOFILE)
		NAMECASE(NPROC)
		NAMECASE(RSS)
		NAMECASE(RTPRIO)
		NAMECASE(SIGPENDING)
		NAMECASE(STACK)
		//NAMECASE(OFILE)
		default: return "UNKNOWN";
	}
}


static const char libtunerlimit_type_envvar[] = "LIBTUNERLIMIT_TYPE";
static const char libtunerlimit_block_envvar[] = "LIBTUNERLIMIT_BLOCK";
static const char libtunerlimit_result_envvar[] = "LIBTUNERLIMIT_RESULT";


static int name_match(const char * name, const char * target) {
	if ( ! strncasecmp(target, "ALL", 3)) {
		return 1;
	} else if ( ! strncasecmp(target, "NONE", 4)) {
		return 0;
	} else if ( ! strcasecmp(name, target)) {
		return 1;
	} else {
		return 0;
	}
}

static int is_match(int resource) {
	const char * name = resource_name(resource);
	char * block;
	char * s;
	char * ch;
	int final;

	//dprintf("libtunerlimit: name = \"%s\"\n", name);
	block = getenv(libtunerlimit_block_envvar);
	if (block == NULL) {
		return 0;
	}
	//dprintf("libtunerlimit: block = \"%s\"\n", block);
	s = ch = block;
	final = 0;
	while (1) {
		if (isspace(*ch) || *ch == ',' || *ch == ';' || *ch == ':' || *ch == '\0') {
			if (*ch == '\0') {
				final = 1;
			} else {
				*ch = '\0';
			}
			//dprintf("libtunerlimit: s = \"%s\"\n", s);
			if (name_match(name, s)) {
				return 1;
			}
			if (final) {
				return 0;
			}
			ch++;
			while (isspace(*ch)) {
				ch++;
			}
			s = ch;
		} else {
			ch++;
		}
	}
	// never reached
	return 0;
}


static int is_dup_only() {
	const char * type = getenv(libtunerlimit_type_envvar);

	if (type == NULL) {
		return 1;
	} else if ( ! strncasecmp(type, "ALL", 3)) {
		return 0;
	} else if ( ! strncasecmp(type, "DUP", 3)) {
		return 1;
	} else {
		return 0;
	}
}


static int is_blocked(int resource, const struct rlimit *rlim) {
	if (is_match(resource)) {
		dprintf("libtunerlimit: resource match found\n");
		if (is_dup_only()) {
			// examine rlim
			struct rlimit actualrlim;
			int rc;
			int errno_orig;
			int myerrno;

			errno_orig = errno;
			errno = 0;
			rc = getrlimit(resource, &actualrlim);
			myerrno = errno;
			errno = errno_orig;
			if (rc != 0) {
				fprintf(stderr, "libtunerlimit: Warning: Unable to get current limits for %s: %s\n", resource_name(resource), strerror(myerrno));
				fprintf(stderr, "libtunerlimit: Warning: allowing call through anyway\n");
				return 0;
			}
			dprintf("libtunerlimit: actual limits: { rlim_cur = %d, rlim_max = %d }\n", actualrlim.rlim_cur, actualrlim.rlim_max);
			if (rlim->rlim_cur == actualrlim.rlim_cur &&
			    rlim->rlim_max == actualrlim.rlim_max) {
				// same, pointless, so blocked
				dprintf("libtunerlimit: idempotent call, blocking\n");
				return 1;
			} else {
				// different, let through
				dprintf("libtunerlimit: non-idempotent call, allowing\n");
				return 0;
			}
		} else {
			dprintf("libtunerlimit: blocking all calls\n");
			return 1;
		}
	} else {
		dprintf("libtunerlimit: resource doesn't match\n");
		return 0;
	}
}



int setrlimit(int resource, const struct rlimit *rlim) {
	static void *libc_handle = NULL;
	static int (*underlying)(int resource, const struct rlimit *rlim) = NULL;
	int retval;
	const char *err;
	const char *result;

	dprintf("libtunerlimit: setrlimit(resource = RLIMIT_%s (%d), %p", resource_name(resource), resource, rlim);
	if (rlim) {
		dprintf(" { rlim_cur = %d, rlim_max = %d }", rlim->rlim_cur, rlim->rlim_max);
	}
	dprintf(")");
	dprintf(": entered\n");

	if (!libc_handle) {
#if defined(RTLD_NEXT)
		libc_handle = RTLD_NEXT;
#else
		libc_handle = dlopen("libc.so.6", RTLD_LAZY);
#endif
		dprintf("libtunerlimit: setrlimit: libc_handle = 0x%x\n", libc_handle);
		if (!libc_handle) {
			fprintf(stderr, "libtunerlimit: Error: Unable to find libc.so: %s\n", dlerror());
			exit(1);
		}

		dlerror();
		underlying = dlsym(libc_handle, "setrlimit");
		dprintf("libtunerlimit: setrlimit: underlying = 0x%x\n", underlying);
		err = dlerror();
		if (err) {
			dprintf("libtunerlimit: setrlimit: err = \"%s\"\n", err);
		}
		if (!underlying || err) {
			fprintf(stderr, "libtunerlimit: Error: Unable to find the underlying setrlimit(): %s\n", dlerror());
			exit(1);
		}
	}

	if (is_blocked(resource, rlim)) {
		result = getenv(libtunerlimit_result_envvar);
		if (result == NULL) {
			retval = -1;
			errno = EPERM;
		} else if ( ! strncasecmp(result, "SUCCESS", 7)) {
			retval = 0;
		} else if ( ! strncasecmp(result, "FAILURE", 7)) {
			retval = -1;
			errno = 0;
		} else if ( ! strncasecmp(result, "FAIL", 4)) {
			retval = -1;
			errno = 0;
		} else if ( ! strncasecmp(result, "EFAULT", 6)) {
			retval = -1;
			errno = EFAULT;
		} else if ( ! strncasecmp(result, "FAULT", 5)) {
			retval = -1;
			errno = EFAULT;
		} else if ( ! strncasecmp(result, "EINVAL", 6)) {
			retval = -1;
			errno = EINVAL;
		} else if ( ! strncasecmp(result, "INVAL", 5)) {
			retval = -1;
			errno = EINVAL;
		} else if ( ! strncasecmp(result, "EPERM", 5)) {
			retval = -1;
			errno = EPERM;
		} else if ( ! strncasecmp(result, "PERM", 4)) {
			retval = -1;
			errno = EPERM;
		} else {
			retval = -1;
			errno = EPERM;
		}
	} else {
		dprintf("libtunerlimit: about to call underlying setrlimit()\n");
		retval = (*underlying)(resource, rlim);
		dprintf("libtunerlimit: underlying setrlimit() = %d\n", retval);
	}

	dprintf("libtunerlimit: setrlimit(resource = RLIMIT_%s (%d), %p", resource_name(resource), resource, rlim);
	if (rlim) {
		dprintf(" { rlim_cur = %d, rlim_max = %d }", rlim->rlim_cur, rlim->rlim_max);
	}
	dprintf(")");
	dprintf(" = %d\n", retval);

	return retval;
}

