/*******************************************************************************
+
+	display.cc
+
+   Copyright (C) 2000
+	Kevin Pulo, kev@hons.cs.usyd.edu.au.
+	Garrick Welsh, gaz@hons.cs.usyd.edu.au.
+
+	This program is free software; you can redistribute it and/or modify
+	it under the terms of the GNU General Public License as published by
+	the Free Software Foundation; either version 2 of the License, or
+	(at your option) any later version.
+
+	This program is distributed in the hope that it will be useful,
+	but WITHOUT ANY WARRANTY; without even the implied warranty of
+	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
+	GNU General Public License for more details.
+
+	You should have received a copy of the GNU General Public License
+	along with this program; if not, write to the Free Software
+	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
+
+	$Id: display.cc,v 1.9 2000/05/28 11:31:19 kev Exp kev $
+
*******************************************************************************/

#include "cluster.hh"


static const char rcsid[] = "$Id: display.cc,v 1.9 2000/05/28 11:31:19 kev Exp kev $";
static const char rcsrevision[] = "$Revision: 1.9 $";
static const char *usage = "Usage: display [options]\n"
"Options:\n"
"    --help, -help, -h  displays this usage message\n"
"    --debug            prints debug information\n"
"    --nodesize         specifies the radius of the displayed nodes\n"
"    --delaunay         toggles showing the delaunay triangulation (default off)\n"
"    --representatives  toggles showing the representative points (default on)\n"
"    --mcedges          toggles showing the M(C) edges (default on)\n"
;



static ofstream null("/dev/null");
ostream *debug;

point_set T;
dictionary<point, string> annotation;
array<Cluster> clusters;
Cluster outliers;


static window *win;

static bool drawDelaunay;
static bool drawRepresentatives;
static bool drawMCEdges;

static int nodesize;


void badCommand(void)
{
	cerr << "Illegal Command" << endl << endl << usage << endl;
	exit(1);
}


/* Skips any spaces and tabs in the stream. */
istream &skipspace(istream &in) {
	int ch;

	ch = in.peek();
	while (ch == ' ' || ch == '\t') {
		in.ignore(1);
		ch = in.peek();
	}

	return in;
}


/* ANSI C++ iostream is useless for reading in everything up to EOL.
   getline() sucks because it needs to be told how large the buffer is,
   and will extract the \n without returning it.  Hence you cannot
   unambiguously know if you've hit the buffer limit or EOL.
   Thus I do it myself.  If you don't like that this method is potentially
   very slow, talk to the ANSI C++ committee, not me.
*/
string readline(istream &in) {
	const int buflen = 1024;
	char s[buflen];
	string res;
	int ch;

	while (true) {
		in.get(s, buflen, '\n');
		res += s;

		ch = in.peek();
		if (ch == EOF) {
			break;
		} else if (ch == '\n') {
			in.ignore(1);
			break;
		}
	}
	return res;
}



void draw_segment(const point& p, const point& q, color c) {
	if (win != NULL) {
		win->draw_segment(p.to_point(), q.to_point(), c);
	}
}

void draw_ray(const point& p, const point& q, color c) {
	if (win != NULL) {
		win->draw_ray(p.to_point(), q.to_point(), c);
	}
}

void draw_edge(const point& p, const point& q, color c) {
	if (win != NULL) {
		win->draw_edge(p.to_point(), q.to_point(), c);
	}
}

void draw_voro_edge(const point& p, const point& q) {
	draw_segment(p, q, blue);
}

void draw_voro_ray(const point& p, const point& q) {
	draw_ray(p, q, blue);
}

void draw_triang_edge(const point& p, const point& q) {
	draw_edge(p, q, yellow);
}

void draw_hull_edge(const point& p, const point& q) {
	draw_edge(p, q, red);
}

void draw_diagram_edge(const point& p, const point& q) {
	draw_edge(p, q, green2);
}

inline color getColor(int i) {
	if (i < 0) {
		return transparent;
	}
	return color(i%14 + 2);
}

void draw(window *win) {
	int i;
	point p;
	int oldLineWidth;

	if (win == NULL) {
		return;
	}

	win->clear();

	// Draw the Delaunay triangulation.
	if (drawDelaunay) {
		(*debug) << "boo!" << endl;
		T.draw_edges(draw_diagram_edge, draw_triang_edge, draw_hull_edge);
	}

	// Draw the outliers.
	list<point> &pts = outliers.getList();
	forall(p, pts) {
		win->draw_filled_node(p, getColor(-1));
	}

	// Draw the clusters.
	for (i = clusters.low(); i <= clusters.high(); i++) {
		Cluster &c = clusters[i];
		list<point> &pts = c.getList();

		// Draw the lines from the points to the representative.
		if (drawMCEdges) {
			oldLineWidth = win->set_line_width(1);
			forall(p, pts) {
				win->draw_segment(p, c.rep, getColor(i));
			}
			win->set_line_width(oldLineWidth);
		}

		// Draw the points themselves.
		forall(p, pts) {
			win->draw_filled_node(p, getColor(i));
		}

		// Draw the representative.
		if (drawRepresentatives) {
			oldLineWidth = win->set_line_width(2);
			win->draw_segment(c.rep.xcoord()-10*nodesize/win->scale(), c.rep.ycoord(), c.rep.xcoord()+10*nodesize/win->scale(), c.rep.ycoord(), getColor(i));
			win->draw_segment(c.rep.xcoord(), c.rep.ycoord()-10*nodesize/win->scale(), c.rep.xcoord(), c.rep.ycoord()+10*nodesize/win->scale(), getColor(i));
			win->set_line_width(oldLineWidth);
		}
	}

	win->redraw_panel();
}

void redraw() {
	draw(win);
}


int main(int argc, char *argv[]) {
	point p;
	string s;
	list<point> L;
	int numClusters = 0;
	double minX = 0.0;
	double minY = 0.0;
	double maxX = 0.0;
	double maxY = 0.0;
	int i;


	// Defaults
	// Non portable, bad form, etc.  Like I care at this point in time.
	debug = &null;
	nodesize = 2;

	drawDelaunay = false;
	drawRepresentatives = true;
	drawMCEdges = true;


	////////////////////
	// Process command line arguments.
	////////////////////
	for (i = 1; i < argc; i++) {
		if ( (string(argv[i]) == "--debug") || (string(argv[i]) == "-debug") ) {
			debug = &cerr;
		} else if ( (string(argv[i]) == "--nodesize") || (string(argv[i]) == "-nodesize") ) {
			i++;
			if (i >= argc) {
				cerr << "ERROR: --nodesize requires an integer parameter." << endl;
				return 1;
			}
			nodesize = atoi(argv[i]);
		} else if ( (string(argv[i]) == "--delaunay") || (string(argv[i]) == "-delaunay") ) {
			drawDelaunay = ! drawDelaunay;
		} else if ( (string(argv[i]) == "--representatives") || (string(argv[i]) == "-representatives") ) {
			drawRepresentatives = ! drawRepresentatives;
		} else if ( (string(argv[i]) == "--mcedges") || (string(argv[i]) == "-mcedges") ) {
			drawMCEdges = ! drawMCEdges;
		} else if ( (string(argv[i]) == "--help") || (string(argv[i]) == "-help") || (string(argv[i]) == "-h") ) {
			cout << usage << endl;
			return 0;
		} else {
			badCommand();
		}
	}


	////////////////////
	// Read the clustered dataset.
	////////////////////
	while (true) {
		// Read neccessary bits.
		cin >> i;
		cin >> p;

		// If any neccesary bits failed, we don't have a complete line,
		// so die.
		if (cin.eof()) {
			break;
		}

		// Read optional bits.
		// EOF in here is fine - incomplete last line, that's all.
		skipspace(cin);
		s = readline(cin);

		// Do stuff with the input.
		if (T.lookup(p) == nil) {
			if (i + 1 > numClusters) {
				numClusters = i + 1;
				(*debug) << "RESIZING CLUSTERS: " << numClusters << endl;
				clusters.resize(numClusters);
			}

			(*debug) << "ADDING: " << i << " " << p << " " << s << endl;
			if (T.points().length() == 0) {
				minX = maxX = p.xcoord();
				minY = maxY = p.ycoord();
			} else {
				if (p.xcoord() < minX) minX = p.xcoord();
				if (p.ycoord() < minY) minY = p.ycoord();
				if (p.xcoord() > maxX) maxX = p.xcoord();
				if (p.ycoord() > maxY) maxY = p.ycoord();
			}
			if (i < 0) {
				outliers.append(p);
				T.insert(p);
				annotation.insert(p, s);
			} else {
				if (s == "representative") {
					clusters[i].rep = p;
				} else {
					T.insert(p);
					annotation.insert(p, s);
					clusters[i].append(p);
				}
			}
		} else {
			cerr << "WARNING: Duplicate point " << p << " ignored." << endl;
		}
	}


	// Write the clustered dataset out (debug).
	for (i = clusters.low(); i <= clusters.high(); i++) {
		Cluster &c = clusters[i];
		list<point> &pts = c.getList();
		forall(p, pts) {
			(*debug) << i << " " << p << " " << annotation.access(p) << endl;
		}
	}


	minX -= minX*0.1;
	minY -= minY*0.1;
	maxX += maxX*0.1;
	maxY += maxY*0.1;

	// Create and display the window.
	float windowHeight;
	float windowWidth;

	windowHeight = 600.0;
	windowWidth = windowHeight/(maxY-minY)*(maxX-minX);
	if (windowWidth > 2*windowHeight) {
		windowWidth = windowHeight;
		windowHeight = windowWidth/(maxX-minX)*(maxY-minY);
	}

	if (NULL == (win = new window(windowWidth, windowHeight, "Display Clusters"))) {
		cerr << "ERROR: Unable to create window." << endl;
		return 1;
	}
	win->init(minX, maxX, minY);
	win->set_redraw(draw);
	win->set_node_width(nodesize);
	win->set_show_coordinates(true);
	win->display();
	redraw();


	// Main loop.
	bool exitRequested;
	bool drawRequested;

	exitRequested = false;
	while(!exitRequested) {
		int evt;
		int val;
		double x, y;

		evt = win->read_event(val, x, y);
		drawRequested = false;
		(*debug) << "GOT EVENT: evt = " << evt << ", val = " << val << ", x = " << x << ", y = " << y << endl;
		switch(evt) {
			case key_press_event:
				switch(val) {
					case 'q':
					case 'Q':
					case KEY_ESCAPE:
						exitRequested = true;
						break;
					case 'D':
					case 'd':
						drawRequested = true;
						drawDelaunay = ! drawDelaunay;
						break;
					case 'R':
					case 'r':
						drawRequested = true;
						drawRepresentatives = ! drawRepresentatives;
						break;
					case 'M':
					case 'm':
						drawRequested = true;
						drawMCEdges = ! drawMCEdges;
						break;
				}
				break;

			case configure_event:
			case exposure_event:
				drawRequested = true;
				break;

			case destroy_event:
				exitRequested = true;
				break;
		}
		if (drawRequested) {
			redraw();
		}

		if (exitRequested) {
			break;
		}

	}

	return 0;
}

